from django.db import models
# from disknet.validators import validate_pdf_extension
from django.core.validators import FileExtensionValidator
from django.contrib.auth.models import User
import uuid
import datetime


# Create your models here.

class Publication(models.Model):
    journal = models.CharField(max_length=200, blank=True, default=None, null=True)
    year = models.IntegerField(blank=True, default=None, null=True)
    volume = models.IntegerField(blank=True, default=None, null=True)
    issue = models.IntegerField(blank=True, default=None, null=True)
    name = models.CharField(max_length=200)
    authors = models.CharField(
        max_length=200, blank=True, default=None, null=True)
    short_cite = models.CharField(
        max_length=200, blank=True, default=None, null=True)
    pages = models.CharField(max_length=200, blank=True, default=None, null=True)
    abstract = models.TextField(blank=True, default=None, null=True)
    note = models.TextField(blank=True, default=None, null=True)
    relevant = models.NullBooleanField(default=None, null=True)
    in_disknet = models.BooleanField(default=False, blank=True, null=True)
    pdf_file = models.FileField(null=True, blank=True, default=None, upload_to="pdfs/%Y/%m/%d",
                                validators=[FileExtensionValidator(['pdf'])])

    assigned_to = models.ForeignKey(
        User, null=True, blank=True, default=None, on_delete=models.SET_NULL)

    def assigned_to_name(self):
        if self.assigned_to:
            return "{} {}".format(self.assigned_to.first_name, self.assigned_to.last_name)
        else:
            return "Not assigned"

    def __str__(self):
        return self.name


class ModelImage(models.Model):
    publication = models.ForeignKey(Publication, on_delete=models.CASCADE)
    image = models.ImageField(null=True, blank=True, default=None, upload_to="model_images/%Y/%m/%d",
                              validators=[FileExtensionValidator(['png', 'jpg', 'jpeg'])])


class Construct(models.Model):
    name = models.CharField(max_length=200)
    definition = models.TextField(null=True, blank=True)

    publication = models.ForeignKey(Publication, on_delete=models.CASCADE)

    def __str__(self):
        return self.name


class Project(models.Model):
    unique_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True, blank=False)
    name = models.CharField(max_length=250)
    description = models.CharField(max_length=250, default="")
    created_at = models.DateTimeField(default=datetime.datetime.now())
    updated_at = models.DateTimeField(default=datetime.datetime.now())
    publications = models.ManyToManyField(Publication, default=None, blank=True, null=True)

    class Meta:
        ordering = ['-updated_at']

    def __str__(self):
        return str(self.name)

    def get_absolute_url(self):
        from django.urls import reverse
        return reverse('disknet:project_detail', args=[str(self.id)])

    # TODO: Meta-Constructs anlegen


class MetaConstruct(models.Model):
    name = models.CharField(max_length=200)
    definition = models.TextField(null=True, blank=True)

    name = models.CharField(max_length=200)

    project = models.ForeignKey(Project, on_delete=models.CASCADE, default=None, blank=True, null=True)
    constructs = models.ManyToManyField(Construct, default=None, blank=True, null=True)

    def __str__(self):
        return self.name


class Item(models.Model):
    text = models.CharField(max_length=250)
    note = models.CharField(max_length=250, default=None, blank=True, null=True)
    reverse_coded = models.NullBooleanField(default=None, null=True)

    publication = models.ForeignKey(Publication, on_delete=models.CASCADE)
    construct = models.ForeignKey(Construct, on_delete=models.CASCADE)

    def __str__(self):
        return self.text


class Relation(models.Model):
    def __str__(self):
        return str(self.id)

    path_coefficient = models.FloatField()

    SIGNIFICANCE_LEVELS = [
        ('<0.001', '<0.001'),
        ('<0.01', '<0.01'),
        ('<0.05', '<0.05'),
        ('<0.1', '<0.1'),
        ('>0.1', '>0.1'),
    ]
    significance = models.CharField(max_length=10, choices=SIGNIFICANCE_LEVELS)
    observations = models.IntegerField()

    publication = models.ForeignKey(
        Publication, related_name='publication', on_delete=models.CASCADE)
    model = models.IntegerField()
    construct_from = models.ForeignKey(
        Construct, related_name='construct_from', on_delete=models.CASCADE, null=True)
    construct_moderator = models.ForeignKey(Construct, related_name='construct_moderator', on_delete=models.CASCADE,
                                            null=True, blank=True, default=None)
    construct_to = models.ForeignKey(
        Construct, related_name='construct_to', on_delete=models.CASCADE, null=True)
