from django.contrib import admin
from django.contrib.admin import AdminSite
from django.contrib.auth.models import Group, User
from django.contrib.auth.admin import GroupAdmin, UserAdmin
from django.contrib.sites.admin import Site, SiteAdmin
from django.urls import resolve
from django import forms

from django.contrib.flatpages.admin import FlatPageAdmin
from django.contrib.flatpages.models import FlatPage
from django.utils.translation import gettext_lazy as _

from django.core import serializers

from django.http import HttpResponseRedirect
from django.contrib.contenttypes.models import ContentType

from django.http import HttpResponse
from django.utils.safestring import mark_safe

from django.http import request
from django.urls import reverse
from django.contrib.admin.templatetags.admin_urls import add_preserved_filters

from .models import Project, Publication, Construct, Relation, Item, ModelImage, MetaConstruct


class MyAdminSite(AdminSite):
    # Text to put in each page's <h1>.
    site_header = 'Disknet administration'

    # Text to put at the top of the admin index page.
    index_title = 'Disknet admin'


admin_site = MyAdminSite(name='myadmin')

hiwi_users = Group.objects.get(name="Hiwis").user_set.all()
managerkit_users = Group.objects.get(name="ManagerKIT").user_set.all()


# Register your models here.


class ConstructInline(admin.TabularInline):
    model = Construct
    extra = 1


class ModelImageInline(admin.StackedInline):
    model = ModelImage
    extra = 1


class MetaConstructInline(admin.StackedInline):
    model = MetaConstruct
    extra = 1


def create_relation_inline_form(parent_publicaiton_id):
    class RelationInlineForm(forms.ModelForm):
        def __init__(self, *args, **kwargs):
            super(RelationInlineForm, self).__init__(*args, **kwargs)

            parent_publication = Publication.objects.get(pk=parent_publicaiton_id)

            for construct_name in ['construct_from', 'construct_to', 'construct_moderator']:
                self.fields[construct_name].queryset = Construct.objects.filter(
                    publication=parent_publication)

        class Meta:
            model = Relation
            fields = ['construct_from', 'construct_moderator', 'construct_to', 'significance', 'model',
                      'path_coefficient', 'observations']

    return RelationInlineForm


def create_relation_inline(parent_publicaiton_id):
    class RelationInline(admin.TabularInline):
        def __init__(self, *args, **kwargs):
            self.model = Relation
            self.extra = 1
            self.form = create_relation_inline_form(parent_publicaiton_id=parent_publicaiton_id)
            super(RelationInline, self).__init__(*args, **kwargs)

    return RelationInline


# def create_meta_construct_inline(project_id):
#     """
#     :param project_id: id of current changing project
#     :return: inline form with all available constructs for this project
#     """
#     class MetaConstructInline(admin.TabularInline):
#         def __init__(self, *args, **kwargs):
#             self.model = MetaConstruct
#             self.extra = 1
#             self.form = create_meta_construct_inline_form(project_id=project_id)
#             super(MetaConstructInline, self).__init__(*args, **kwargs)
#
#     return MetaConstructInline

# def create_meta_construct_inline_form(project_id):
#     class MetaConstructInlineForm(forms.ModelForm):
#         def __init__(self, *args, **kwargs):
#             super(MetaConstructInlineForm, self).__init__(*args, **kwargs)
#
#             project = Project.objects.get(pk=project_id)
#             project_publications = project.publications.all()
#
#             # TODO: Next: unterscheidung zwischen From/Mod/To Construkten zur erstellung von Metakonstrukten beruecksichtigen
#
#             for construct_name in ['construct_from', 'construct_to', 'construct_moderator']:
#                 self.fields[construct_name].queryset = Construct.objects.filter(
#                     publication=parent_publication)
#
#         class Meta:
#             model = Relation
#             fields = ['construct_from', 'construct_moderator', 'construct_to', 'significance', 'model',
#                       'path_coefficient', 'observations']
#
#     return RelationInlineForm

class PublicationAdmin(admin.ModelAdmin):
    """
    custom publication admin has a custom view for adding and changing publications.
    """
    # A template for a customized change and add view:
    change_form_template = 'admin/publication_change_form.html'
    add_form_template = 'admin/publication_add_form.html'

    list_per_page = 100

    def __init__(self, *args, **kwargs):
        super(PublicationAdmin, self).__init__(*args, **kwargs)

    def get_osm_info(self):
        # ...
        pass

    def pdf_file_link(self, instance):
        if instance.pdf_file:
            return mark_safe("<a href='%s'>PDF</a>" % (instance.pdf_file.url,))
        else:
            return "No PDF"

    pdf_file_link.allow_tags = True

    def make_relevant(self, request, queryset):
        rows_updated = queryset.update(relevant=True)
        if rows_updated == 1:
            message_bit = "1 publication was"
        else:
            message_bit = "%s publications were" % rows_updated
        self.message_user(request, "%s successfully marked as relevant." % message_bit)

    make_relevant.short_description = "Mark selection as relevant"

    def make_not_relevant(self, request, queryset):
        rows_updated = queryset.update(relevant=False)
        if rows_updated == 1:
            message_bit = "1 publication was"
        else:
            message_bit = "%s publications were" % rows_updated
        self.message_user(request, "%s successfully marked as not relevant." % message_bit)

    make_not_relevant.short_description = "Mark selection as not relevant"

    def make_assign_to_user_action(self, user):
        def assign_to_user(self, request, queryset):
            rows_updated = queryset.update(assigned_to=user.id)
            if rows_updated == 1:
                message_bit = "1 publication was"
            else:
                message_bit = "%s publications were" % rows_updated
            self.message_user(request, "{} successfully assigned to {} {}.".format(message_bit, user.first_name,
                                                                                   user.last_name))

        assign_to_user.short_description = "Assign to {0} {1}".format(user.first_name, user.last_name)
        # We need a different '__name__' for each action - Django
        # uses this as a key in the drop-down box.
        assign_to_user.__name__ = 'assign_to_user_{0}'.format(user.id)

        return assign_to_user

    def make_assign_to_project(self, project):
        """
        :param project: selected project in dropdown
        :return: formatted project-assign action for admin dropdown
        """

        def assign_to_project(self, request, queryset):
            selected_project = Project.objects.get(name=project.name)
            rows_updated = 0
            for publication in queryset:
                if publication not in selected_project.publications.all():
                    selected_project.publications.add(publication)
                    rows_updated += 1

            if rows_updated == 1:
                message_bit = "1 publication was"
            else:
                message_bit = "%s publications were" % rows_updated
            self.message_user(request, "{} successfully assigned to project {}.".format(message_bit, project.name))

        assign_to_project.short_description = "Assign to project {}".format(project.name)
        # We need a different '__name__' for each action - Django
        # uses this as a key in the drop-down box.
        assign_to_project.__name__ = 'assign_to_project_{}'.format(project.name)

        return assign_to_project

    def remove_assignment(self, request, queryset):
        rows_updated = queryset.update(assigned_to=None)
        if rows_updated == 1:
            message_bit = "1 publication assignment was"
        else:
            message_bit = "%s publication assignments were" % rows_updated
        self.message_user(request, "%s successfully removed." % message_bit)

    remove_assignment.short_description = "Remove selected publication assignments"

    def get_actions(self, request):
        actions = super().get_actions(request)
        if request.user in hiwi_users:
            # Hiwis have no actions available

            actions = None
            return actions

        for user in User.objects.all().order_by('first_name'):
            action = self.make_assign_to_user_action(user)
            actions[action.__name__] = (action,
                                        action.__name__,
                                        action.short_description)

        if request.user in managerkit_users:
            # projects actions only for KIT managers available (beta feature)
            for project in Project.objects.all().order_by('name'):
                action = self.make_assign_to_project(project)
                actions[action.__name__] = (action,
                                            action.__name__,
                                            action.short_description)
        return actions

    def add_view(self, request, form_url='', extra_context=None):
        self.inlines = [ModelImageInline, ConstructInline]

        if request.user in hiwi_users:
            # exclude some fields for hiwis
            self.exclude = ('assigned_to', 'relevant',)

        return super().add_view(request, form_url)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        """

        :param request:
        :param object_id:
        :param form_url:
        :param extra_context:
        :return: change add relations view with filtered constructs - show only constructs for this publication
        """

        relation_instance = create_relation_inline(parent_publicaiton_id=object_id)

        self.inlines = [ModelImageInline, ConstructInline, relation_instance]

        if request.user in hiwi_users:
            # exclude some fields for hiwis
            self.exclude = ('assigned_to', 'relevant')

        extra_context = extra_context or {}
        extra_context['osm_data'] = self.get_osm_info()

        return super().change_view(
            request, object_id, form_url, extra_context=extra_context,
        )

    list_display = (
        'name', 'short_cite', 'authors', 'year', 'journal', "pdf_file_link", "relevant", 'in_disknet',
        "assigned_to_name")

    def get_list_filter(self, request):
        if request.user in managerkit_users:
            # project filter only for KIT managers available in (beta feature)
            list_filter = ['year', 'journal', 'relevant', 'in_disknet', 'assigned_to', 'project']
        else:
            list_filter = ['year', 'journal', 'relevant', 'in_disknet', 'assigned_to']
        return list_filter

    search_fields = ['name', 'short_cite', 'authors']
    actions = ['assign_to_user', 'remove_assignment', 'make_relevant', 'make_not_relevant']


class ConstructAdmin(admin.ModelAdmin):
    fieldsets = [
        (None, {'fields': ['publication']}),
        (None, {'fields': ['name']}),
        (None, {'fields': ['definition']})]


class RelationAdmin(admin.ModelAdmin):
    fieldsets = [
        (None, {'fields': ['publication']}),
        (None, {'fields': ['construct_from']}),
        (None, {'fields': ['construct_moderator']}),
        (None, {'fields': ['model']}),
        (None, {'fields': ['construct_to']}),
        (None, {'fields': ['path_coefficient']}),
        (None, {'fields': ['observations']}),
        (None, {'fields': ['significance']})]


class ProjectAdmin(admin.ModelAdmin):
    fieldsets = [
        (None, {'fields': ['name']}),
        (None, {'fields': ['publications']}),
    ]
    filter_horizontal = ('publications',)

    # meta_construct_inline = create_relation_inline(parent_publicaiton_id=object_id)

    # def change_view(self, request, object_id, form_url='', extra_context=None):
    #     meta_construct_inline = create_meta_construct_inline(project_id=object_id)
    #     self.inlines = [meta_construct_inline]
    #
    #     return super().change_view(request, object_id, form_url, extra_context=extra_context,)

    view_on_site = True


# Define a new FlatPageAdmin
class FlatPageAdmin(FlatPageAdmin):
    fieldsets = (
        (None, {'fields': ('url', 'title', 'content', 'sites')}),
        (_('Advanced options'), {
            'classes': ('collapse',),
            'fields': (
                'enable_comments',
                'registration_required',
                'template_name',
            ),
        }),
    )


# register FlatPageAdmin
admin_site.register(FlatPage, FlatPageAdmin)

admin_site.register(Site, SiteAdmin)

admin_site.register(Project, ProjectAdmin)
admin_site.register(Publication, PublicationAdmin)
# admin_site.register(Construct, ConstructAdmin)
# admin_site.register(Relation, RelationAdmin)
admin_site.register(Group, GroupAdmin)
admin_site.register(User, UserAdmin)
